using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		/// Data access layer component implementing data persistence logic for supply. 
		/// Exposes public static methods allowing retrieve and save supplies data.
		///</summary>

		#endregion

	public class Supply
	{
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1962"> 
		///		<ExpectedInput>Divsion Code, Supply Type Id, Manufacturer Name, Lot Number</ExpectedInput>
		///		<ExpectedOutput>DataTable with supplies</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1963"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a list of supplies from database based on search criteria.
		/// </summary>
		/// <param name="supplyType">Supply type id</param>
		/// <param name="manufacturerName">Manufacturer name</param>
		/// <param name="lotNumber">Lot number</param>
		/// <param name="includeAll">Option to include expired/unsatisfactory supplies</param>
		/// <returns>DataTable containing list of supplies satisying the search criteria</returns>
		public static DataTable GetSupplies(Common.SupplyType supplyType, string manufacturerName, string lotNumber, bool includeAll)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSupplies.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetSupplies.supplytypeid, System.Data.SqlDbType.Int),
				new SqlParameter(STOREDPROC.GetSupplies.manufacturername, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetSupplies.lotnumber, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetSupplies.includeall, System.Data.SqlDbType.Bit)
			};
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			
			if (supplyType == Common.SupplyType.AllSupplies)
			{
				prms[1].Value = null;
			}
			else
			{
				prms[1].Value = (int)supplyType;
			}
			
			if ((manufacturerName==null)||(manufacturerName.Trim().Length == 0))
			{
				prms[2].Value = null;
			}
			else
			{
				prms[2].Value = manufacturerName;
			}
			
			if ((lotNumber==null)||(lotNumber.Trim().Length == 0))
			{
				prms[3].Value = null;
			}
			else
			{
				prms[3].Value = lotNumber;
			}
			
			prms[4].Value = includeAll;

			DataSet dsResults = Common.StoredProcedure.GetData(STOREDPROC.GetSupplies.StoredProcName, prms);
			
			return dsResults.Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1339"> 
		///		<ExpectedInput>Invoice#, SupplyId, Lotnumber</ExpectedInput>
		///		<ExpectedOutput>DataTable with Supply Data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1340"> 
		///		<ExpectedInput>INvalid Parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets supply information if found
		/// </summary>
		/// <param name="invoiceNumber">Invoice to search on</param>
		/// <param name="lotNumber">Lot number to search on</param>
		/// <param name="supplyTypeId">Supply Type Id to serach on</param>
		/// <returns></returns>
		public static DataTable GetSupplyByInvoiceLotNumber(string invoiceNumber, string lotNumber, int supplyTypeId)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSupplyByInvoiceLotNumber.divisioncode, SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetSupplyByInvoiceLotNumber.invoicenumber, SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetSupplyByInvoiceLotNumber.lotnumber, SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetSupplyByInvoiceLotNumber.supplytypeid, SqlDbType.Int)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[1].Value = invoiceNumber;
			prms[2].Value = lotNumber;
			prms[3].Value = supplyTypeId;

			return (Common.StoredProcedure.GetDataTable(STOREDPROC.GetSupplyByInvoiceLotNumber.StoredProcName, prms));
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4135"> 
		///		<ExpectedInput>SupplyGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of Supply Information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4136"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a supply record
		/// </summary>
		/// <param name="supplyGuid"></param>
		/// <returns></returns>
		public static DataTable GetSupplyByGuid(System.Guid supplyGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSupplyByGuid.supplyguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetSupplyByGuid.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = supplyGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetSupplyByGuid.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4137"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable shema for Supply</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4138"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a blank Supply DataTable
		/// </summary>
		/// <returns></returns>
		public static DataTable GetEmptySupplyTable()
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(TABLE.Supply.TableName);

			dtEmpty.Columns.Add(TABLE.Supply.Comments, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.DivisionCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.SupplyGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.Supply.SupplyName, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.SatisfactoryIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.Supply.AvailableQuantity, typeof(int));
			dtEmpty.Columns.Add(TABLE.Supply.SupplyTypeId, typeof(int));
			dtEmpty.Columns.Add(TABLE.Supply.InvoiceNumber, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.LotExpirationDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.Supply.ManufacturerName, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.ReceiptDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.Supply.ReceiptTechId, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.LastUpdateDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.Supply.LastUpdateUser, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.LastUpdateFunctionId, typeof(int));
			dtEmpty.Columns.Add(TABLE.Supply.RowVersion, typeof(byte[]));
			dtEmpty.Columns.Add(TABLE.Supply.LotNumber, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.LotQuantity, typeof(short));
			dtEmpty.Columns.Add(TABLE.Supply.InventoryChangeDate, typeof(System.DateTime));

			dtEmpty.Columns.Add(TABLE.SupplyType.SupplyTypeText, typeof(string));
			return(dtEmpty);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2053"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable shema for spInsertSupply</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2054"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a empty DataTable
		/// </summary>
		/// <returns></returns>
		public static DataTable GetInsertSupplyDataTable()
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(TABLE.Supply.TableName);

			dtEmpty.Columns.Add(TABLE.Supply.SupplyGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.Supply.SupplyTypeId, typeof(int));
			dtEmpty.Columns.Add(TABLE.Supply.SupplyName, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.ManufacturerName, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.ReceiptDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.Supply.ReceiptTechId, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.InvoiceNumber, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.LotNumber, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.LotExpirationDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.Supply.LotQuantity, typeof(int));
			dtEmpty.Columns.Add(TABLE.Supply.SatisfactoryIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.Supply.AvailableQuantity, typeof(int));
			dtEmpty.Columns.Add(TABLE.Supply.Comments, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.DivisionCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.Supply.LastUpdateUser, typeof(string));

			dtEmpty.Columns.Add(TABLE.Supply.LastUpdateFunctionId, typeof(int));
			
			return(dtEmpty);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1737"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1738"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Grabs a DataTable of unique ManufacturerNames from the Supply table
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetManufacturerNameList()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSupplyManufacturerList.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetSupplyManufacturerList.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1741"> 
		///		<ExpectedInput>Valid manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Data Table of supply names</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1742"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Grabs a DataTable of unique SupplyNames from the Supply table for a given manufacturer
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetSupplyNameList(string manufacturerName)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSupplyNameList.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetSupplyNameList.manufacturername, System.Data.SqlDbType.VarChar)
			};
			//
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			//
			if (manufacturerName.Trim().Length>0)
			{
				prms[1].Value = manufacturerName;
			}
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetSupplyNameList.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1745"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1748"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Grabs a DataTable of SupplyTypeId, SupplyTypeText from the SupplyType table
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetSupplyTypeList()
		{
			return Common.StoredProcedure.GetData(STOREDPROC.GetSupplyTypeList.StoredProcName).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1749"> 
		///		<ExpectedInput>DataTable of Supply Records to Insert</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1750"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inserts rows from input DataTable into the Supply table
		/// </summary>
		/// <returns></returns>
		public static bool InsertSupplies(System.Data.DataTable dtSupplies, System.Data.DataTable dtWorkloadEvents)
		{
			DataTable[] dtArray = new DataTable[2];
			string[] spArray = new string[2];
			//
			dtArray[0] = dtSupplies;
			spArray[0] = STOREDPROC.InsertSupply.StoredProcName;
			//
			dtArray[1] = dtWorkloadEvents;
			spArray[1] = STOREDPROC.InsertWorkloadEvents.StoredProcName;

			
			return (new StoredProcedure().TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1822"> 
		///		<ExpectedInput>DataTables, Update value</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1823"> 
		///		<ExpectedInput>Invalid DataTables</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates supply record
		/// </summary>
		/// <param name="dtSupply">Update values for Supply record</param>
		/// <param name="dtWorkload">Workload generated by the supply update</param>
		/// <param name="lastUpdateFunctionId">Which function is updating the data</param>
		/// <returns>Boolean</returns>
		public static bool UpdateSupplyUsage(DataTable dtSupply, DataTable dtWorkload, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList sprocArray = new System.Collections.ArrayList();

			//Add the LastUpdateFunctionId to the tables, and append to the collectionlist
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtSupply, lastUpdateFunctionId));
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtWorkload, lastUpdateFunctionId));

			sprocArray.Add(STOREDPROC.UpdateSupplyUsage.StoredProcName);
			sprocArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);

			int retValue = new StoredProcedure().TransactionalGetValue(sprocArray, dtArray);

			return (retValue == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1787"> 
		///		<ExpectedInput>Parameter values for report</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1788"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetSupplyInventoryReportData
		/// </summary>
		/// <param name="receiptStartDate"></param>
		/// <param name="receiptEndDate"></param>
		/// <param name="expirationStartDate"></param>
		/// <param name="expirationEndDate"></param>
		/// <param name="includeUnsatisfactory"></param>
		/// <param name="supplyTypeIds"></param>
		/// <returns></returns>
		public static DataTable GetSupplyInventoryReportData(System.DateTime receiptStartDate, System.DateTime receiptEndDate,
															 System.DateTime expirationStartDate, System.DateTime expirationEndDate,
															 bool includeUnsatisfactory, string supplyTypeIds)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSupplyInventoryReportData.divisioncode, SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetSupplyInventoryReportData.expirationenddate, SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetSupplyInventoryReportData.expirationstartdate, SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetSupplyInventoryReportData.includeunsatisfactory, SqlDbType.Bit),
				new SqlParameter(STOREDPROC.GetSupplyInventoryReportData.receiptenddate, SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetSupplyInventoryReportData.receiptstartdate, SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetSupplyInventoryReportData.supplytypeid, SqlDbType.VarChar),

			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			//.NET and sQL have different MaxValues/MinValues
			if (expirationEndDate == System.DateTime.MaxValue)
				prms[1].Value = DBNull.Value;
			else
                prms[1].Value = expirationEndDate;

			//.NET and sQL have different MaxValues/MinValues
			if (expirationStartDate == System.DateTime.MinValue)
				prms[2].Value = DBNull.Value;
			else
				prms[2].Value = expirationStartDate;
			
			prms[3].Value = includeUnsatisfactory;
			
			//.NET and sQL have different MaxValues/MinValues
			if (receiptEndDate == System.DateTime.MaxValue)
				prms[4].Value = DBNull.Value;
			else
				prms[4].Value = receiptEndDate;

			//.NET and sQL have different MaxValues/MinValues
			if (receiptStartDate == System.DateTime.MinValue)
				prms[5].Value = DBNull.Value;
			else
				prms[5].Value = receiptStartDate;

			//Which Supply Type's are requested?
			prms[6].Value = supplyTypeIds;

			return Common.StoredProcedure.GetData(STOREDPROC.GetSupplyInventoryReportData.StoredProcName, prms).Tables[0];

		}

	}
}





